<?php

namespace App\Models;

use CodeIgniter\Model;

class DepositsMadeModel extends Model
{
    protected $DBGroup = 'default';

    // We'll not define $table because we'll manually insert into
    // either table: 'deposits_made' or 'deposits_made_lines'.

    /**
     * Upsert a single Deposit object from QBO into
     * `deposits_made` + `deposits_made_lines`.
     */
    public function upsertDeposit(array $depositData): void
    {
        // 1) Basic fields
        $depId  = $depositData['Id']       ?? null;
        if (!$depId) {
            return; // skip if no QBO ID
        }
        $depDate      = $depositData['TxnDate']    ?? null;
        $docNumber    = $depositData['DocNumber']  ?? '';
        $depToAcctRef = $depositData['DepositToAccountRef']['value'] ?? '';
        $depToAcctNam = $depositData['DepositToAccountRef']['name']  ?? '';
        $depTotal     = $depositData['TotalAmt']   ?? 0;
        $depMemo      = $depositData['PrivateNote'] ?? '';

        // 2) Insert or update the deposit header
        // We'll see if there's an existing row with qb_deposit_id = ...
        $db = db_connect();

        // find existing row in deposits_made
        $existing = $db->table('deposits_made')
            ->where('qb_deposit_id', $depId)
            ->get()
            ->getRowArray();

        $headerData = [
            'qb_deposit_id'                => $depId,
            'deposit_date'                 => $depDate,
            'doc_number'                   => $docNumber,
            'deposit_to_account_ref_value' => $depToAcctRef,
            'deposit_to_account_ref_name'  => $depToAcctNam,
            'total_amt'                    => $depTotal,
            'memo'                         => $depMemo,
        ];

        if ($existing) {
            // update
            $db->table('deposits_made')
               ->where('id', $existing['id'])
               ->update($headerData);

            $headerId = $existing['id'];
        } else {
            // insert
            $db->table('deposits_made')->insert($headerData);
            $headerId = $db->insertID();
        }

        // 3) Now handle deposit lines
        // We’ll remove old lines for this deposit then insert fresh lines
        $db->table('deposits_made_lines')
           ->where('deposits_made_id', $headerId)
           ->delete();

        foreach ($depositData['Line'] ?? [] as $line) {
            $lineNum     = $line['Id']              ?? '';
            $lineDesc    = $line['Description']     ?? '';
            $lineAmount  = $line['Amount']          ?? 0;
            // The deposit line might link to multiple Txns
            $linkedArray = $line['LinkedTxn']       ?? [];

            // If there are no LinkedTxns, still store a line
            if (empty($linkedArray)) {
                $this->insertLineRow($db, $headerId, $lineNum, $lineDesc, $lineAmount, '', '');
            } else {
                // For each LinkedTxn, insert a new row
                foreach ($linkedArray as $lt) {
                    $this->insertLineRow(
                        $db,
                        $headerId,
                        $lineNum,
                        $lineDesc,
                        $lineAmount,
                        ($lt['TxnType'] ?? ''),
                        ($lt['TxnId']   ?? '')
                    );
                }
            }
        }
    }

    /**
     * Insert a single row into deposits_made_lines.
     */
    private function insertLineRow(
        $db,
        int $headerId,
        string $lineNum,
        string $description,
        float $amount,
        string $txnType,
        string $txnId
    ): void
    {
        $linesTable = $db->table('deposits_made_lines');
        $linesTable->insert([
            'deposits_made_id' => $headerId,
            'line_num'         => $lineNum,
            'description'      => $description,
            'amount'           => $amount,
            'linked_txn_type'  => $txnType,
            'linked_txn_id'    => $txnId,
        ]);
    }
}
