<?php

namespace App\Controllers;

use CodeIgniter\Controller;
use App\Models\UndepositedFundsModel;

class UndepositedFunds extends Controller
{
    /** Helper to check if string contains "Undeposited" */
    private function strContainsUndeposited(?string $s): bool
    {
        return $s && stripos($s, 'undeposited') !== false;
    }

    public function index()
    {
        // DEV mode: ensure fresh .env
        if (ENVIRONMENT === 'development') {
            cache()->clean();
        }

        // Auth gate
        if (! service('auth')->loggedIn()) {
            return redirect()->to('/login');
        }
        $user = service('auth')->user();
        if ((int) $user->is_admin !== 1) {
            return redirect()->to('/')->with('error', 'Access denied: Admins only.');
        }

        // Session QBO check
        $session     = session();
        $realmId     = $session->get('realmid');
        $accessToken = $session->get('access_token');
        $isConnected = ($realmId && $accessToken);

        // Form inputs
        $req       = service('request');
        $fromDate  = $req->getGetPost('from_date');
        $toDate    = $req->getGetPost('to_date');
        $action    = $req->getPost('action');

        // Default date range
        $firstOfMonth = date('Y-m-01');
        $today        = date('Y-m-d');
        if (empty($fromDate)) {
            $fromDate = $firstOfMonth;
        }
        if (empty($toDate)) {
            $toDate = $today;
        }

        // Pull data if requested
        if ($action === 'pull_undeposited' && $isConnected && $fromDate && $toDate) {
            try {
                $ufModel = new UndepositedFundsModel();
                $this->pullPayments($realmId, $accessToken, $fromDate, $toDate, $ufModel);
                $this->pullSalesReceipts($realmId, $accessToken, $fromDate, $toDate, $ufModel);
                $this->pullRefundReceipts($realmId, $accessToken, $fromDate, $toDate, $ufModel);
                $this->pullCreditMemos($realmId, $accessToken, $fromDate, $toDate, $ufModel);
                $this->pullJournalEntries($realmId, $accessToken, $fromDate, $toDate, $ufModel);

                // Also pull deposit data
                $this->pullDeposits($realmId, $accessToken, $fromDate, $toDate);
                $this->crossReferenceDeposits();
            } catch (\Throwable $e) {
                $error = 'Error while pulling data: ' . $e->getMessage();
            }
        }
        if ($action === 'pull_deposits' && $isConnected && $fromDate && $toDate) {
            try {
                $this->pullDeposits($realmId, $accessToken, $fromDate, $toDate);
                $this->crossReferenceDeposits();
            } catch (\Throwable $e) {
                $error = 'Error while pulling deposits: ' . $e->getMessage();
            }
        }

        // Fetch results
        $ufModel  = new UndepositedFundsModel();
        $limitVal = $req->getGetPost('limit') ?: '100';
        $all      = ($limitVal === 'ALL');
        $results  = [];

        if ($all) {
            // Show all records in the date range
            $results = $ufModel->where('txn_date >=', $fromDate)
                               ->where('txn_date <=', $toDate)
                               ->orderBy('txn_date', 'ASC')
                               ->findAll();
            $pager = null; // no pagination
        } else {
            // Use pagination
            $perPage = (int) $limitVal;
            if ($perPage < 1) {
                $perPage = 100; // fallback
            }
            // You can chain "where(...)" or write a custom method. We'll do a quick approach:
            $ufModel->where('txn_date >=', $fromDate)
                    ->where('txn_date <=', $toDate)
                    ->orderBy('txn_date', 'ASC');

            $results = $ufModel->paginate($perPage, 'ufGroup'); // 'ufGroup' is the pager group
            $pager   = $ufModel->pager;
        }

        // Prepare data
        $data = [
            'isConnected' => $isConnected,
            'error'       => $error ?? null,
            'from_date'   => $fromDate,
            'to_date'     => $toDate,
            'results'     => $results,
            'pager'       => $pager,
            'limitVal'    => $limitVal
        ];

        return view('undepositedfunds/index', $data);
    }

    /**
     * loadView: partial HTML based on ?view_mode
     * /undepositedfunds/loadView?view_mode=make_bank_deposit
     */
    public function loadView()
    {
        $req   = service('request');
        $view  = $req->getGet('view_mode');
        $session     = session();
        $realmId     = $session->get('realmid');
        $accessToken = $session->get('access_token');
        $isConnected = ($realmId && $accessToken);

        // We can still read limit if needed
        $limitVal = $req->getGet('limit') ?: '100';
        $all      = ($limitVal === 'ALL');
        $pager    = null;

        // For date-based views, we read from/to
        $fromDate = $req->getGet('from_date') ?? date('Y-m-01');
        $toDate   = $req->getGet('to_date')   ?? date('Y-m-d');

        $data = [
            'isConnected' => $isConnected,
            'error'       => '',
            'from_date'   => $fromDate,
            'to_date'     => $toDate,
            'limitVal'    => $limitVal,
            'pager'       => null,
            'results'     => [],
        ];

        // date-selector partial
        if ($view === '_date_selector') {
            return view('undepositedfunds/left/_date_selector', $data);
        }

        // "pull" partial
        if ($view === 'pull') {
            return view('undepositedfunds/middle/pull_undeposited_funds', $data);
        }

        // "make_bank_deposit" partial => only is_deposited=0
        if ($view === 'make_bank_deposit') {
            $ufModel = new UndepositedFundsModel();
            if ($all) {
                $results = $ufModel->where('is_deposited', 0)
                                   ->orderBy('txn_date', 'ASC')
                                   ->findAll();
            } else {
                $perPage = (int) $limitVal;
                if ($perPage < 1) {
                    $perPage = 100;
                }
                $ufModel->where('is_deposited', 0)->orderBy('txn_date', 'ASC');
                $results = $ufModel->paginate($perPage, 'mbdGroup');
                $pager   = $ufModel->pager;
            }
            $data['results'] = $results;
            $data['pager']   = $pager;

            return view('undepositedfunds/middle/make_bank_deposit', $data);
        }

        // default "manage" partial
        $ufModel = new UndepositedFundsModel();
        if ($all) {
            $results = $ufModel->where('txn_date >=', $fromDate)
                               ->where('txn_date <=', $toDate)
                               ->orderBy('txn_date', 'ASC')
                               ->findAll();
        } else {
            $perPage = (int) $limitVal;
            if ($perPage < 1) {
                $perPage = 100;
            }
            $ufModel->where('txn_date >=', $fromDate)
                    ->where('txn_date <=', $toDate)
                    ->orderBy('txn_date', 'ASC');
            $results = $ufModel->paginate($perPage, 'ufGroup');
            $pager   = $ufModel->pager;
        }

        $data['results'] = $results;
        $data['pager']   = $pager;

        return view('undepositedfunds/middle/manage_undeposited_funds', $data);
    }

    /* ------------------------ PULL METHODS ------------------------ */

    private function pullDeposits(string $realm, string $tok, string $from, string $to): void
    {
        $ufModel = new UndepositedFundsModel();
        $start   = 1;
        $max     = 1000;
        do {
            $q = "
              SELECT *
              FROM Deposit
              WHERE TxnDate >= '{$from}' AND TxnDate <= '{$to}'
              ORDERBY Id
              STARTPOSITION {$start}
              MAXRESULTS {$max}";
            $rows = $this->runQuery($realm, $tok, $q)['QueryResponse']['Deposit'] ?? [];
            foreach ($rows as $d) {
                $ufModel->upsertDeposit($d);
            }
            $start += $max;
        } while (count($rows) === $max);
    }

    private function pullPayments(string $realm, string $tok, string $from, string $to, UndepositedFundsModel $ufModel)
    {
        $start = 1;
        $max   = 1000;
        do {
            $q = "
              SELECT *
              FROM Payment
              WHERE TxnDate >= '{$from}' AND TxnDate <= '{$to}'
              ORDERBY Id
              STARTPOSITION {$start}
              MAXRESULTS {$max}";
            $rows = $this->runQuery($realm, $tok, $q)['QueryResponse']['Payment'] ?? [];
            foreach ($rows as $p) {
                $ufModel->upsertUndepositedRecord($p, 'Payment');
            }
            $start += $max;
        } while (count($rows) === $max);
    }

    private function pullSalesReceipts(string $realm, string $tok, string $from, string $to, UndepositedFundsModel $ufModel)
    {
        $start = 1;
        $max   = 1000;
        do {
            $q = "
              SELECT *
              FROM SalesReceipt
              WHERE TxnDate >= '{$from}' AND TxnDate <= '{$to}'
              ORDERBY Id
              STARTPOSITION {$start}
              MAXRESULTS {$max}";
            $rows = $this->runQuery($realm, $tok, $q)['QueryResponse']['SalesReceipt'] ?? [];
            foreach ($rows as $sr) {
                $acctName = $sr['DepositToAccountRef']['name'] ?? '';
                if ($this->strContainsUndeposited($acctName)) {
                    $ufModel->upsertUndepositedRecord($sr, 'SalesReceipt');
                }
            }
            $start += $max;
        } while (count($rows) === $max);
    }

    private function pullRefundReceipts(string $realm, string $tok, string $from, string $to, UndepositedFundsModel $ufModel)
    {
        $start = 1;
        $max   = 1000;
        do {
            $q = "
              SELECT *
              FROM RefundReceipt
              WHERE TxnDate >= '{$from}' AND TxnDate <= '{$to}'
              ORDERBY Id
              STARTPOSITION {$start}
              MAXRESULTS {$max}";
            $rows = $this->runQuery($realm, $tok, $q)['QueryResponse']['RefundReceipt'] ?? [];
            foreach ($rows as $rr) {
                $acctName = $rr['RefundFromAccountRef']['name']
                            ?? $rr['DepositToAccountRef']['name']
                            ?? '';
                if ($this->strContainsUndeposited($acctName)) {
                    $ufModel->upsertUndepositedRecord($rr, 'RefundReceipt');
                }
            }
            $start += $max;
        } while (count($rows) === $max);
    }

    private function pullCreditMemos(string $realm, string $tok, string $from, string $to, UndepositedFundsModel $ufModel)
    {
        $start = 1;
        $max   = 1000;
        do {
            $q = "
              SELECT *
              FROM CreditMemo
              WHERE TxnDate >= '{$from}' AND TxnDate <= '{$to}'
              ORDERBY Id
              STARTPOSITION {$start}
              MAXRESULTS {$max}";
            $rows = $this->runQuery($realm, $tok, $q)['QueryResponse']['CreditMemo'] ?? [];
            foreach ($rows as $cm) {
                $acctName = $cm['DepositToAccountRef']['name'] ?? '';
                if ($this->strContainsUndeposited($acctName)) {
                    $ufModel->upsertUndepositedRecord($cm, 'CreditMemo');
                }
            }
            $start += $max;
        } while (count($rows) === $max);
    }

    private function pullJournalEntries(string $realm, string $tok, string $from, string $to, UndepositedFundsModel $ufModel)
    {
        $start = 1;
        $max   = 1000;
        do {
            $q = "
              SELECT *
              FROM JournalEntry
              WHERE TxnDate >= '{$from}' AND TxnDate <= '{$to}'
              ORDERBY Id
              STARTPOSITION {$start}
              MAXRESULTS {$max}";
            $rows = $this->runQuery($realm, $tok, $q)['QueryResponse']['JournalEntry'] ?? [];
            foreach ($rows as $je) {
                $hasUF = false;
                foreach ($je['Line'] ?? [] as $ln) {
                    $ref = $ln['JournalEntryLineDetail']['AccountRef'] ?? [];
                    if ($this->strContainsUndeposited($ref['name'] ?? '')) {
                        $hasUF = true;
                        break;
                    }
                }
                if ($hasUF) {
                    $ufModel->upsertUndepositedRecord($je, 'JournalEntry');
                }
            }
            $start += $max;
        } while (count($rows) === $max);
    }

    /** Helper to run QBO queries */
    private function runQuery(string $realm, string $tok, string $q): array
    {
        $url = "https://quickbooks.api.intuit.com/v3/company/{$realm}/query?query=" . urlencode($q);

        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_HTTPHEADER     => [
                "Authorization: Bearer {$tok}",
                "Accept: application/json",
                "Content-Type: application/text",
            ],
            CURLOPT_RETURNTRANSFER => true,
        ]);
        $resp = curl_exec($ch);
        $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($code !== 200) {
            throw new \RuntimeException("QBO API error {$code}: {$resp}");
        }
        return json_decode($resp, true);
    }

    /** Cross-reference deposit lines to update is_deposited, deposit_txn_id, deposit_date */
    private function crossReferenceDeposits(): void
    {
        $db = \Config\Database::connect();
        $sql = "
            UPDATE undeposited_funds uf
            JOIN deposits_made_lines dml ON dml.linked_txn_id = uf.qb_txn_id
            JOIN deposits_made       dm  ON dm.id            = dml.deposits_made_id
            SET uf.is_deposited    = 1,
                uf.deposit_txn_id  = dm.qb_deposit_id,
                uf.deposit_date    = dm.deposit_date
            WHERE uf.is_deposited  = 0
        ";
        $db->query($sql);
    }
}
