<?php

namespace App\Controllers;

use CodeIgniter\Controller;
use App\Models\UndepositedFundsModel;

class UndepositedFunds extends Controller
{
    private function strContainsUndeposited(?string $s): bool
    {
        return $s && stripos($s, 'undeposited') !== false;
    }

    public function index()
    {
        if (ENVIRONMENT === 'development') {
            cache()->clean();
        }

        if (! service('auth')->loggedIn()) {
            return redirect()->to('/login');
        }
        $user = service('auth')->user();
        if ((int) $user->is_admin !== 1) {
            return redirect()->to('/')->with('error', 'Access denied: Admins only.');
        }

        $session     = session();
        $realmId     = $session->get('realmid');
        $accessToken = $session->get('access_token');
        $isConnected = ($realmId && $accessToken);

        $req      = service('request');
        $fromDate = $req->getGetPost('from_date');
        $toDate   = $req->getGetPost('to_date');
        $action   = $req->getPost('action');

        $firstOfMonth = date('Y-m-01');
        $today        = date('Y-m-d');
        if (empty($fromDate)) {
            $fromDate = $firstOfMonth;
        }
        if (empty($toDate)) {
            $toDate = $today;
        }

        // Pull data if requested
        if ($action === 'pull_undeposited' && $isConnected && $fromDate && $toDate) {
            try {
                $ufModel = new UndepositedFundsModel();
                $this->pullPayments($realmId, $accessToken, $fromDate, $toDate, $ufModel);
                $this->pullSalesReceipts($realmId, $accessToken, $fromDate, $toDate, $ufModel);
                $this->pullRefundReceipts($realmId, $accessToken, $fromDate, $toDate, $ufModel);
                $this->pullCreditMemos($realmId, $accessToken, $fromDate, $toDate, $ufModel);
                $this->pullJournalEntries($realmId, $accessToken, $fromDate, $toDate, $ufModel);

                $this->pullDeposits($realmId, $accessToken, $fromDate, $toDate);
                $this->crossReferenceDeposits();
            } catch (\Throwable $e) {
                $error = 'Error while pulling data: ' . $e->getMessage();
            }
        }
        if ($action === 'pull_deposits' && $isConnected && $fromDate && $toDate) {
            try {
                $this->pullDeposits($realmId, $accessToken, $fromDate, $toDate);
                $this->crossReferenceDeposits();
            } catch (\Throwable $e) {
                $error = 'Error while pulling deposits: ' . $e->getMessage();
            }
        }

        $ufModel  = new UndepositedFundsModel();
        $limitVal = $req->getGetPost('limit') ?: '100';
        $all      = ($limitVal === 'ALL');
        $results  = [];
        $pager    = null;

        if ($all) {
            $results = $ufModel->where('txn_date >=', $fromDate)
                               ->where('txn_date <=', $toDate)
                               ->orderBy('txn_date', 'ASC')
                               ->findAll();
        } else {
            $perPage = (int) $limitVal;
            if ($perPage < 1) {
                $perPage = 100;
            }
            $ufModel->where('txn_date >=', $fromDate)
                    ->where('txn_date <=', $toDate)
                    ->orderBy('txn_date', 'ASC');
            $results = $ufModel->paginate($perPage, 'ufGroup');
            $pager   = $ufModel->pager;
        }

        $data = [
            'isConnected' => $isConnected,
            'error'       => $error ?? null,
            'from_date'   => $fromDate,
            'to_date'     => $toDate,
            'results'     => $results,
            'pager'       => $pager,
            'limitVal'    => $limitVal
        ];

        return view('undepositedfunds/index', $data);
    }

    public function loadView()
    {
        $req      = service('request');
        $view     = $req->getGet('view_mode');
        $fresh    = $req->getGet('fresh') ?? '0'; // '1' means clear session
        $session  = session();

        $realmId     = $session->get('realmid');
        $accessToken = $session->get('access_token');
        $isConnected = ($realmId && $accessToken);

        $limitVal = $req->getGet('limit') ?: '100';
        $all      = ($limitVal === 'ALL');
        $pager    = null;

        $fromDate = $req->getGet('from_date') ?? date('Y-m-01');
        $toDate   = $req->getGet('to_date')   ?? date('Y-m-d');

        // If user is NOT going to "make_bank_deposit", remove existing session array
        if ($view !== 'make_bank_deposit') {
            $session->remove('transaction_ids_selected');
        }

        $data = [
            'isConnected' => $isConnected,
            'error'       => '',
            'from_date'   => $fromDate,
            'to_date'     => $toDate,
            'limitVal'    => $limitVal,
            'pager'       => null,
            'results'     => [],
        ];

        // partials
        if ($view === '_date_selector') {
            return view('undepositedfunds/left/_date_selector', $data);
        }
        if ($view === 'pull') {
            return view('undepositedfunds/middle/pull_undeposited_funds', $data);
        }

        // If user is going to "make_bank_deposit"
        if ($view === 'make_bank_deposit') {
            // If fresh=1 => remove old selections
            if ($fresh === '1') {
                $session->remove('transaction_ids_selected');
            }

            $ufModel = new UndepositedFundsModel();
            if ($all) {
                $results = $ufModel->where('is_deposited', 0)
                                   ->orderBy('txn_date', 'ASC')
                                   ->findAll();
            } else {
                $perPage = (int) $limitVal;
                if ($perPage < 1) {
                    $perPage = 100;
                }
                $ufModel->where('is_deposited', 0)->orderBy('txn_date', 'ASC');
                $results = $ufModel->paginate($perPage, 'mbdGroup');
                $pager   = $ufModel->pager;
            }
            $data['results'] = $results;
            $data['pager']   = $pager;

            return view('undepositedfunds/middle/make_bank_deposit', $data);
        }

        // Otherwise default "manage"
        $ufModel = new UndepositedFundsModel();
        if ($all) {
            $results = $ufModel->where('txn_date >=', $fromDate)
                               ->where('txn_date <=', $toDate)
                               ->orderBy('txn_date', 'ASC')
                               ->findAll();
        } else {
            $perPage = (int) $limitVal;
            if ($perPage < 1) {
                $perPage = 100;
            }
            $ufModel->where('txn_date >=', $fromDate)
                    ->where('txn_date <=', $toDate)
                    ->orderBy('txn_date', 'ASC');
            $results = $ufModel->paginate($perPage, 'ufGroup');
            $pager   = $ufModel->pager;
        }

        $data['results'] = $results;
        $data['pager']   = $pager;

        return view('undepositedfunds/middle/manage_undeposited_funds', $data);
    }

    // ---------------------
    //  Pull Methods (same)
    // ---------------------
    private function pullDeposits(string $realm, string $tok, string $from, string $to): void
    {
        $ufModel = new UndepositedFundsModel();
        $start   = 1;
        $max     = 1000;
        do {
            $q = "
              SELECT *
              FROM Deposit
              WHERE TxnDate >= '{$from}' AND TxnDate <= '{$to}'
              ORDERBY Id
              STARTPOSITION {$start}
              MAXRESULTS {$max}";
            $rows = $this->runQuery($realm, $tok, $q)['QueryResponse']['Deposit'] ?? [];
            foreach ($rows as $d) {
                $ufModel->upsertDeposit($d);
            }
            $start += $max;
        } while (count($rows) === $max);
    }

    private function pullPayments(string $realm, string $tok, string $from, string $to, UndepositedFundsModel $ufModel)
    {
        $start = 1;
        $max   = 1000;
        do {
            $q = "
              SELECT *
              FROM Payment
              WHERE TxnDate >= '{$from}' AND TxnDate <= '{$to}'
              ORDERBY Id
              STARTPOSITION {$start}
              MAXRESULTS {$max}";
            $rows = $this->runQuery($realm, $tok, $q)['QueryResponse']['Payment'] ?? [];
            foreach ($rows as $p) {
                $ufModel->upsertUndepositedRecord($p, 'Payment');
            }
            $start += $max;
        } while (count($rows) === $max);
    }

    private function pullSalesReceipts(string $realm, string $tok, string $from, string $to, UndepositedFundsModel $ufModel)
    {
        $start = 1;
        $max   = 1000;
        do {
            $q = "
              SELECT *
              FROM SalesReceipt
              WHERE TxnDate >= '{$from}' AND TxnDate <= '{$to}'
              ORDERBY Id
              STARTPOSITION {$start}
              MAXRESULTS {$max}";
            $rows = $this->runQuery($realm, $tok, $q)['QueryResponse']['SalesReceipt'] ?? [];
            foreach ($rows as $sr) {
                $acctName = $sr['DepositToAccountRef']['name'] ?? '';
                if ($this->strContainsUndeposited($acctName)) {
                    $ufModel->upsertUndepositedRecord($sr, 'SalesReceipt');
                }
            }
            $start += $max;
        } while (count($rows) === $max);
    }

    private function pullRefundReceipts(string $realm, string $tok, string $from, string $to, UndepositedFundsModel $ufModel)
    {
        $start = 1;
        $max   = 1000;
        do {
            $q = "
              SELECT *
              FROM RefundReceipt
              WHERE TxnDate >= '{$from}' AND TxnDate <= '{$to}'
              ORDERBY Id
              STARTPOSITION {$start}
              MAXRESULTS {$max}";
            $rows = $this->runQuery($realm, $tok, $q)['QueryResponse']['RefundReceipt'] ?? [];
            foreach ($rows as $rr) {
                $acctName = $rr['RefundFromAccountRef']['name']
                            ?? $rr['DepositToAccountRef']['name']
                            ?? '';
                if ($this->strContainsUndeposited($acctName)) {
                    $ufModel->upsertUndepositedRecord($rr, 'RefundReceipt');
                }
            }
            $start += $max;
        } while (count($rows) === $max);
    }

    private function pullCreditMemos(string $realm, string $tok, string $from, string $to, UndepositedFundsModel $ufModel)
    {
        $start = 1;
        $max   = 1000;
        do {
            $q = "
              SELECT *
              FROM CreditMemo
              WHERE TxnDate >= '{$from}' AND TxnDate <= '{$to}'
              ORDERBY Id
              STARTPOSITION {$start}
              MAXRESULTS {$max}";
            $rows = $this->runQuery($realm, $tok, $q)['QueryResponse']['CreditMemo'] ?? [];
            foreach ($rows as $cm) {
                $acctName = $cm['DepositToAccountRef']['name'] ?? '';
                if ($this->strContainsUndeposited($acctName)) {
                    $ufModel->upsertUndepositedRecord($cm, 'CreditMemo');
                }
            }
            $start += $max;
        } while (count($rows) === $max);
    }

    private function pullJournalEntries(string $realm, string $tok, string $from, string $to, UndepositedFundsModel $ufModel)
    {
        $start = 1;
        $max   = 1000;
        do {
            $q = "
              SELECT *
              FROM JournalEntry
              WHERE TxnDate >= '{$from}' AND TxnDate <= '{$to}'
              ORDERBY Id
              STARTPOSITION {$start}
              MAXRESULTS {$max}";
            $rows = $this->runQuery($realm, $tok, $q)['QueryResponse']['JournalEntry'] ?? [];
            foreach ($rows as $je) {
                $hasUF = false;
                foreach ($je['Line'] ?? [] as $ln) {
                    $ref = $ln['JournalEntryLineDetail']['AccountRef'] ?? [];
                    if ($this->strContainsUndeposited($ref['name'] ?? '')) {
                        $hasUF = true;
                        break;
                    }
                }
                if ($hasUF) {
                    $ufModel->upsertUndepositedRecord($je, 'JournalEntry');
                }
            }
            $start += $max;
        } while (count($rows) === $max);
    }

    private function runQuery(string $realm, string $tok, string $q): array
    {
        $url = "https://quickbooks.api.intuit.com/v3/company/{$realm}/query?query=" . urlencode($q);

        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_HTTPHEADER     => [
                "Authorization: Bearer {$tok}",
                "Accept: application/json",
                "Content-Type: application/text",
            ],
            CURLOPT_RETURNTRANSFER => true,
        ]);
        $resp = curl_exec($ch);
        $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($code !== 200) {
            throw new \RuntimeException("QBO API error {$code}: {$resp}");
        }
        return json_decode($resp, true);
    }

    private function crossReferenceDeposits(): void
    {
        $db = \Config\Database::connect();
        $sql = "
            UPDATE undeposited_funds uf
            JOIN deposits_made_lines dml ON dml.linked_txn_id = uf.qb_txn_id
            JOIN deposits_made       dm  ON dm.id            = dml.deposits_made_id
            SET uf.is_deposited    = 1,
                uf.deposit_txn_id  = dm.qb_deposit_id,
                uf.deposit_date    = dm.deposit_date
            WHERE uf.is_deposited  = 0
        ";
        $db->query($sql);
    }

    // ---------------------------------------------------
    // updateSelected() now only updates the session data
    // ---------------------------------------------------
    public function updateSelected()
    {
        if (! $this->request->isAJAX()) {
            return $this->response->setJSON(['error' => 'AJAX only'])->setStatusCode(400);
        }
        $body = $this->request->getJSON(true);
        $ufId    = $body['ufId']    ?? null;
        $checked = $body['checked'] ?? false;

        if (! $ufId) {
            return $this->response->setJSON(['error' => 'No ufId'])->setStatusCode(400);
        }

        $session = session();
        $selected = $session->get('transaction_ids_selected') ?? [];

        if ($checked) {
            if (! in_array($ufId, $selected)) {
                $selected[] = $ufId;
            }
        } else {
            $selected = array_filter($selected, function ($id) use ($ufId) {
                return $id != $ufId;
            });
        }
        $session->set('transaction_ids_selected', $selected);

        // No sum logic now, just return success
        return $this->response->setJSON(['success' => true]);
    }

    // ---------------------------------------------------------
    // NEW: Serve the left-pane "Make Bank Deposit" view directly
    // ---------------------------------------------------------
    public function leftMakeBankDeposit()
    {
        return view('undepositedfunds/left/make_bank_deposit');
    }

    // ---------------------------------------------------------
    // NEW: Clear all selected transaction IDs, and return main left-pane
    // ---------------------------------------------------------
    public function clearSelected()
    {
        session()->remove('transaction_ids_selected');
        return view('undepositedfunds/left/main_view_selector');
    }
}
