<?php

namespace App\Controllers;

use CodeIgniter\Controller;
use App\Models\UndepositedFundsModel;

class UndepositedFunds extends Controller
{
    /** quick helper so we do the “contains Undeposited” test in one place */
    private function strContainsUndeposited(?string $s): bool
    {
        return $s && stripos($s, 'undeposited') !== false;
    }

    public function index()
    {
        //------------------------------------------------------------------
        // 0) ensure we’re always reading the *latest* .env (dev only)
        //------------------------------------------------------------------
        if (ENVIRONMENT === 'development') {
            // clears cached config & .env values in CI4
            cache()->clean();
        }

        //------------------------------------------------------------------
        // 1) auth gate
        //------------------------------------------------------------------
        if (! service('auth')->loggedIn()) {
            return redirect()->to('/login');
        }
        $user = service('auth')->user();
        if ((int) $user->is_admin !== 1) {
            return redirect()->to('/')->with('error', 'Access denied: Admins only.');
        }

        //------------------------------------------------------------------
        // 2) QBO connection in session?
        //------------------------------------------------------------------
        $session     = session();
        $realmId     = $session->get('realmid');
        $accessToken = $session->get('access_token');
        $isConnected = ($realmId && $accessToken);

        //------------------------------------------------------------------
        // 3) read form & set defaults
        //------------------------------------------------------------------
        $req      = service('request');
        $fromDate = $req->getGetPost('from_date');
        $toDate   = $req->getGetPost('to_date');
        $action   = $req->getPost('action');

        // If user did not provide from_date or to_date, default to:
        //   first day of current month --> today
        $firstOfMonth = date('Y-m-01');
        $today        = date('Y-m-d');

        if (empty($fromDate)) {
            $fromDate = $firstOfMonth;
        }
        if (empty($toDate)) {
            $toDate = $today;
        }

        //------------------------------------------------------------------
        // 4) view vars
        //------------------------------------------------------------------
        $data = [
            'isConnected' => $isConnected,
            'error'       => null,
            'from_date'   => $fromDate,
            'to_date'     => $toDate,
            'results'     => [],
        ];

        //------------------------------------------------------------------
        // 5) pull undeposited-funds set
        //------------------------------------------------------------------
        if ($action === 'pull_undeposited' && $isConnected && $fromDate && $toDate) {
            try {
                $ufModel = new UndepositedFundsModel();

                // Pull QuickBooks objects that might be in Undeposited Funds
                $this->pullPayments($realmId, $accessToken, $fromDate, $toDate, $ufModel);
                $this->pullSalesReceipts($realmId, $accessToken, $fromDate, $toDate, $ufModel);
                $this->pullRefundReceipts($realmId, $accessToken, $fromDate, $toDate, $ufModel);
                $this->pullCreditMemos($realmId, $accessToken, $fromDate, $toDate, $ufModel);
                $this->pullJournalEntries($realmId, $accessToken, $fromDate, $toDate, $ufModel);

                // Also pull deposit data here so the user doesn't have to click twice
                $this->pullDeposits($realmId, $accessToken, $fromDate, $toDate);

                // Now cross-reference and update deposit flags
                $this->crossReferenceDeposits();
            } catch (\Throwable $e) {
                $data['error'] = 'Error while pulling data: ' . $e->getMessage();
            }
        }

        //------------------------------------------------------------------
        // 6) separate block for pulling deposits only
        //------------------------------------------------------------------
        if ($action === 'pull_deposits' && $isConnected && $fromDate && $toDate) {
            try {
                $this->pullDeposits($realmId, $accessToken, $fromDate, $toDate);

                // Cross-reference after deposit pull
                $this->crossReferenceDeposits();
            } catch (\Throwable $e) {
                $data['error'] = 'Error while pulling deposits: ' . $e->getMessage();
            }
        }

        //------------------------------------------------------------------
        // 7) load grid rows
        //------------------------------------------------------------------
        $data['results'] = (new UndepositedFundsModel())->fetchByDateRange($fromDate, $toDate);

        //------------------------------------------------------------------
        // 8) render
        //------------------------------------------------------------------
        return view('undepositedfunds/index', $data);
    }

    /**
     * loadView method: returns partial HTML based on `view_mode`, or `_date_selector`.
     * Called by AJAX: /undepositedfunds/loadView?view_mode=_date_selector
     */
    public function loadView()
    {
        $req      = service('request');
        $viewMode = $req->getGet('view_mode');

        // Provide the same defaults that 'index()' does:
        $session     = session();
        $realmId     = $session->get('realmid');
        $accessToken = $session->get('access_token');
        $isConnected = ($realmId && $accessToken);

        $fromDate  = $req->getGet('from_date') ?? date('Y-m-01');
        $toDate    = $req->getGet('to_date')   ?? date('Y-m-d');

        $data = [
            'isConnected' => $isConnected,
            'error'       => '',
            'from_date'   => $fromDate,
            'to_date'     => $toDate,
            'results'     => [],
        ];

        // Additional date-selector partial
        if ($viewMode === '_date_selector') {
            return view('undepositedfunds/left/_date_selector', $data);
        }

        // "pull" partial
        if ($viewMode === 'pull') {
            return view('undepositedfunds/middle/pull_undeposited_funds', $data);
        }

        // default to "manage" partial
        return view('undepositedfunds/middle/manage_undeposited_funds', $data);
    }

    /* ──────────────────────────────────────────────────────────────
       Deposit header/line puller
       ──────────────────────────────────────────────────────────── */
    private function pullDeposits(string $realm, string $tok, string $from, string $to): void
    {
        $ufModel = new UndepositedFundsModel();
        $start   = 1;
        $max     = 1000;

        do {
            $q = "
              SELECT *
              FROM Deposit
              WHERE TxnDate >= '{$from}' AND TxnDate <= '{$to}'
              ORDERBY Id
              STARTPOSITION {$start}
              MAXRESULTS {$max}";
            $rows = $this->runQuery($realm, $tok, $q)['QueryResponse']['Deposit'] ?? [];
            foreach ($rows as $d) {
                $ufModel->upsertDeposit($d);
            }
            $start += $max;
        } while (count($rows) === $max);
    }

    /* ──────────────────────────────────────────────────────────────
       Payment
       ──────────────────────────────────────────────────────────── */
    private function pullPayments(string $realm, string $tok, string $from, string $to, UndepositedFundsModel $ufModel)
    {
        $start = 1;
        $max   = 1000;

        do {
            $q = "
              SELECT *
              FROM Payment
              WHERE TxnDate >= '{$from}' AND TxnDate <= '{$to}'
              ORDERBY Id
              STARTPOSITION {$start}
              MAXRESULTS {$max}";
            $rows = $this->runQuery($realm, $tok, $q)['QueryResponse']['Payment'] ?? [];

            foreach ($rows as $p) {
                $ufModel->upsertUndepositedRecord($p, 'Payment');
            }

            $start += $max;
        } while (count($rows) === $max);
    }

    /* ──────────────────────────────────────────────────────────────
       SalesReceipt
       ──────────────────────────────────────────────────────────── */
    private function pullSalesReceipts(string $realm, string $tok, string $from, string $to, UndepositedFundsModel $ufModel)
    {
        $start = 1;
        $max   = 1000;
        do {
            $q = "
              SELECT *
              FROM SalesReceipt
              WHERE TxnDate >= '{$from}' AND TxnDate <= '{$to}'
              ORDERBY Id
              STARTPOSITION {$start}
              MAXRESULTS {$max}";
            $rows = $this->runQuery($realm, $tok, $q)['QueryResponse']['SalesReceipt'] ?? [];

            foreach ($rows as $sr) {
                $acctName = $sr['DepositToAccountRef']['name'] ?? '';
                if ($this->strContainsUndeposited($acctName)) {
                    $ufModel->upsertUndepositedRecord($sr, 'SalesReceipt');
                }
            }
            $start += $max;
        } while (count($rows) === $max);
    }

    /* ──────────────────────────────────────────────────────────────
       RefundReceipt
       ──────────────────────────────────────────────────────────── */
    private function pullRefundReceipts(string $realm, string $tok, string $from, string $to, UndepositedFundsModel $ufModel)
    {
        $start = 1;
        $max   = 1000;
        do {
            $q = "
              SELECT *
              FROM RefundReceipt
              WHERE TxnDate >= '{$from}' AND TxnDate <= '{$to}'
              ORDERBY Id
              STARTPOSITION {$start}
              MAXRESULTS {$max}";
            $rows = $this->runQuery($realm, $tok, $q)['QueryResponse']['RefundReceipt'] ?? [];

            foreach ($rows as $rr) {
                $acctName = $rr['RefundFromAccountRef']['name']
                            ?? $rr['DepositToAccountRef']['name']
                            ?? '';
                if ($this->strContainsUndeposited($acctName)) {
                    $ufModel->upsertUndepositedRecord($rr, 'RefundReceipt');
                }
            }
            $start += $max;
        } while (count($rows) === $max);
    }

    /* ──────────────────────────────────────────────────────────────
       CreditMemo
       ──────────────────────────────────────────────────────────── */
    private function pullCreditMemos(string $realm, string $tok, string $from, string $to, UndepositedFundsModel $ufModel)
    {
        $start = 1;
        $max   = 1000;
        do {
            $q = "
              SELECT *
              FROM CreditMemo
              WHERE TxnDate >= '{$from}' AND TxnDate <= '{$to}'
              ORDERBY Id
              STARTPOSITION {$start}
              MAXRESULTS {$max}";
            $rows = $this->runQuery($realm, $tok, $q)['QueryResponse']['CreditMemo'] ?? [];

            foreach ($rows as $cm) {
                $acctName = $cm['DepositToAccountRef']['name'] ?? '';
                if ($this->strContainsUndeposited($acctName)) {
                    $ufModel->upsertUndepositedRecord($cm, 'CreditMemo');
                }
            }
            $start += $max;
        } while (count($rows) === $max);
    }

    /* ──────────────────────────────────────────────────────────────
       JournalEntry
       ──────────────────────────────────────────────────────────── */
    private function pullJournalEntries(string $realm, string $tok, string $from, string $to, UndepositedFundsModel $ufModel)
    {
        $start = 1;
        $max   = 1000;

        do {
            $q = "
              SELECT *
              FROM JournalEntry
              WHERE TxnDate >= '{$from}' AND TxnDate <= '{$to}'
              ORDERBY Id
              STARTPOSITION {$start}
              MAXRESULTS {$max}";
            $rows = $this->runQuery($realm, $tok, $q)['QueryResponse']['JournalEntry'] ?? [];

            foreach ($rows as $je) {
                $hasUF = false;
                foreach ($je['Line'] ?? [] as $ln) {
                    $ref = $ln['JournalEntryLineDetail']['AccountRef'] ?? [];
                    if ($this->strContainsUndeposited($ref['name'] ?? '')) {
                        $hasUF = true;
                        break;
                    }
                }
                if ($hasUF) {
                    $ufModel->upsertUndepositedRecord($je, 'JournalEntry');
                }
            }
            $start += $max;
        } while (count($rows) === $max);
    }

    /* ──────────────────────────────────────────────────────────────
       helper to run queries
       ──────────────────────────────────────────────────────────── */
    private function runQuery(string $realm, string $tok, string $q): array
    {
        $url = "https://quickbooks.api.intuit.com/v3/company/{$realm}/query?query=" . urlencode($q);

        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_HTTPHEADER     => [
                "Authorization: Bearer {$tok}",
                "Accept: application/json",
                "Content-Type: application/text",
            ],
            CURLOPT_RETURNTRANSFER => true,
        ]);
        $resp = curl_exec($ch);
        $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($code !== 200) {
            throw new \RuntimeException("QBO API error {$code}: {$resp}");
        }
        return json_decode($resp, true);
    }

    /* ──────────────────────────────────────────────────────────────
       Cross-reference deposits
       ──────────────────────────────────────────────────────────── */
    private function crossReferenceDeposits(): void
    {
        $db = \Config\Database::connect();
        $sql = "
            UPDATE undeposited_funds uf
            JOIN   deposits_made_lines dml ON dml.linked_txn_id = uf.qb_txn_id
            JOIN   deposits_made       dm  ON dm.id            = dml.deposits_made_id
            SET    uf.is_deposited    = 1,
                   uf.deposit_txn_id  = dm.qb_deposit_id,
                   uf.deposit_date    = dm.deposit_date
            WHERE  uf.is_deposited    = 0
        ";
        $db->query($sql);
    }
}
