<?php

namespace App\Controllers;

use CodeIgniter\Controller;
use App\Models\MenuModel;
use App\Models\CommissionOverrideModel;

/**
 * CommissionSettings
 *
 * Shows the page with commission overrides,
 * plus any notes or links to other "pull" endpoints.
 */
class CommissionSettings extends Controller
{
    public function index()
    {
        // 1. Must be logged in
        if (! service('auth')->loggedIn()) {
            return redirect()->to('/login');
        }

        // 2. Must be admin
        $user = service('auth')->user();
        if ((int) $user->is_admin !== 1) {
            return redirect()->to('/')->with('error', 'Access denied: Admins only.');
        }

        // 3. Get nav menu
        $menuModel = new MenuModel();
        $menuItems = $menuModel->orderBy('order', 'ASC')->findAll();

        // 4. Get commission override data
        $commissionModel = new CommissionOverrideModel();
        $commissionData = $commissionModel
            ->select('commission_override.*, bbt_users.first_name, bbt_users.last_name')
            ->join('bbt_users', 'commission_override.user_id = bbt_users.id', 'left')
            ->findAll();

        // 5. Show the view
        return view('commissionsettings/index', [
            'menuItems'      => $menuItems,
            'commissionData' => $commissionData
        ]);
    }

    /**
     * Updates all commission overrides at once.
     */
    public function updateAll()
    {
        // 1. Must be logged in
        if (! service('auth')->loggedIn()) {
            return redirect()->to('/login');
        }

        // 2. Must be admin
        $user = service('auth')->user();
        if ((int) $user->is_admin !== 1) {
            return redirect()->to('/')->with('error', 'Access denied: Admins only.');
        }

        // 3. CommissionOverrideModel
        $commissionModel = new CommissionOverrideModel();

        // 4. Gather posted arrays
        $deposit            = $this->request->getPost('deposit');
        $customer_care      = $this->request->getPost('customer_care');
        $title_work         = $this->request->getPost('title_work');
        $checkout_completed = $this->request->getPost('checkout_completed');
        $upsells            = $this->request->getPost('upsells');

        if (! is_array($deposit)) {
            return redirect()->back()->with('error', 'No data to update.');
        }

        // 5. Loop each row
        foreach ($deposit as $id => $val) {
            $data = [
                'deposit'            => $deposit[$id]            ?? 0,
                'customer_care'      => $customer_care[$id]      ?? 0,
                'title_work'         => $title_work[$id]         ?? 0,
                'checkout_completed' => $checkout_completed[$id] ?? 0,
                'upsells'            => $upsells[$id]            ?? 0,
            ];
            $commissionModel->update($id, $data);
        }

        return redirect()->back()->with('success', 'All rows updated successfully!');
    }
}
