<?php

namespace App\Models;

use CodeIgniter\Model;

/**
 * Combined model that handles:
 *  - Undeposited Funds (in `undeposited_funds` table)
 *  - Deposits (in `deposits_made` and `deposits_made_lines` tables)
 *
 * All logic from both UndepositedFundsModel and DepositsMadeModel is placed here.
 */
class UndepositedFundsModel extends Model
{
    // -----------------------------------------------------------------------
    // Undeposited Funds portion
    // -----------------------------------------------------------------------
    protected $table      = 'undeposited_funds';
    protected $primaryKey = 'id';

    protected $allowedFields = [
        'qb_txn_id',
        'txn_type',
        'txn_date',
        'customer_name',
        'payment_method',
        'amount',
        'deposit_to_account',
        'memo',
        'reference_number',
        'is_deposited',
        'deposit_txn_id',
        'deposit_date',
        'journal_reference',
        'created_at',
        // JournalEntry extras
        'debit_amount',
        'credit_amount',
        'je_description',
        'je_account_name',
        'je_line_name',
        // Additional columns
        'payment_ref_num',
        'refund_from_account',
        'unapplied_amt',
        'exchange_rate',
        'currency_ref_name',
        'ar_account_ref_name',
        'line_items_json',
    ];

    /**
     * Upsert a transaction record in `undeposited_funds`
     * for Payment | SalesReceipt | RefundReceipt | CreditMemo | JournalEntry.
     */
    public function upsertUndepositedRecord(array $txn, string $txnType): void
    {
        $qbId = $txn['Id'] ?? null;
        if (! $qbId) {
            return;
        }

        // Common fields for all transaction types
        $data = [
            'qb_txn_id'        => $qbId,
            'txn_type'         => $txnType,
            'txn_date'         => $txn['TxnDate']            ?? null,
            'memo'             => $txn['PrivateNote']        ?? '',
            'reference_number' => $txn['DocNumber']
                                   ?? ($txn['TxnReferenceNumber'] ?? ''),
            'customer_name'    => $txn['CustomerRef']['name'] ?? '',
            'payment_method'   => $txn['PaymentMethodRef']['name'] ?? '',
            'amount'           => $txn['TotalAmt']           ?? 0,
            'is_deposited'     => 0,
        ];

        // Multi-currency
        if (! empty($txn['ExchangeRate'])) {
            $data['exchange_rate'] = $txn['ExchangeRate'];
        }
        if (! empty($txn['CurrencyRef']['name'])) {
            $data['currency_ref_name'] = $txn['CurrencyRef']['name'];
        }

        switch ($txnType) {
            // ----------------------------------------------------------
            // 1) Payment (Receive Payment)
            // ----------------------------------------------------------
            case 'Payment':
                $data['payment_ref_num'] = $txn['PaymentRefNum'] ?? '';
                $data['unapplied_amt']   = $txn['UnappliedAmt']  ?? 0;

                if (! empty($txn['ARAccountRef']['name'])) {
                    $data['ar_account_ref_name'] = $txn['ARAccountRef']['name'];
                }
                if (! empty($txn['Line'])) {
                    $data['line_items_json'] = json_encode($txn['Line']);
                }

                // “Undeposited Funds” may be in DepositToAccountRef or BankAccountRef.
                $depName  = $txn['DepositToAccountRef']['name'] ?? '';
                $bankName = $txn['BankAccountRef']['name']      ?? '';

                if ($depName === '' && $bankName === '') {
                    // Some payments omit both refs — treat them as Undeposited Funds.
                    $data['deposit_to_account'] = 'Undeposited Funds';
                } elseif (stripos($depName, 'Undeposited') !== false) {
                    $data['deposit_to_account'] = $depName;
                } elseif (stripos($bankName, 'Undeposited') !== false) {
                    $data['deposit_to_account'] = $bankName;
                } else {
                    // Not an undeposited payment
                    return;
                }

                // If deposit_to_account is not EXACTLY "Undeposited Funds", skip.
                if ($data['deposit_to_account'] !== 'Undeposited Funds') {
                    return;
                }
                break;

            // ----------------------------------------------------------
            // 2) SalesReceipt
            // ----------------------------------------------------------
            case 'SalesReceipt':
                if (! empty($txn['Line'])) {
                    $data['line_items_json'] = json_encode($txn['Line']);
                }
                $depName = $txn['DepositToAccountRef']['name'] ?? '';
                if (stripos($depName, 'Undeposited') === false) {
                    return;
                }
                $data['deposit_to_account'] = $depName;
                break;

            // ----------------------------------------------------------
            // 3) RefundReceipt
            // ----------------------------------------------------------
            case 'RefundReceipt':
                $refundAcct = $txn['RefundFromAccountRef']['name'] ?? '';
                $depName    = $txn['DepositToAccountRef']['name']  ?? '';

                if (stripos($refundAcct, 'Undeposited') !== false) {
                    $data['refund_from_account'] = $refundAcct;
                    $data['deposit_to_account']  = $refundAcct;
                } elseif (stripos($depName, 'Undeposited') !== false) {
                    $data['refund_from_account'] = $depName;
                    $data['deposit_to_account']  = $depName;
                } else {
                    return;
                }

                if (! empty($txn['Line'])) {
                    $data['line_items_json'] = json_encode($txn['Line']);
                }
                break;

            // ----------------------------------------------------------
            // 4) CreditMemo
            // ----------------------------------------------------------
            case 'CreditMemo':
                $depName = $txn['DepositToAccountRef']['name'] ?? '';
                if (stripos($depName, 'Undeposited') === false) {
                    return;
                }
                $data['deposit_to_account'] = $depName;

                if (! empty($txn['Line'])) {
                    $data['line_items_json'] = json_encode($txn['Line']);
                }
                break;

            // ----------------------------------------------------------
            // 5) JournalEntry
            // ----------------------------------------------------------
            case 'JournalEntry':
                $data['journal_reference'] = $txn['DocNumber'] ?? '';
                $debit  = 0.0;
                $credit = 0.0;
                $desc   = '';
                $acct   = '';
                $name   = '';

                foreach ($txn['Line'] ?? [] as $ln) {
                    $det = $ln['JournalEntryLineDetail'] ?? null;
                    if (! $det) {
                        continue;
                    }
                    $posting  = $det['PostingType']        ?? '';
                    $amt      = (float) ($ln['Amount']     ?? 0);
                    $acctNm   = $det['AccountRef']['name'] ?? '';
                    $lineDesc = $ln['Description']         ?? '';
                    $entity   = $det['Entity']['EntityRef']['name']
                                ?? ($det['Entity']['Name'] ?? '');

                    if (stripos($acctNm, 'Undeposited Funds') !== false) {
                        if ($posting === 'Debit')  { $debit  += $amt; }
                        if ($posting === 'Credit') { $credit += $amt; }

                        $desc = $desc ?: $lineDesc;
                        $acct = $acct ?: $acctNm;
                        $name = $name ?: $entity;
                    }
                }

                if ($debit == 0 && $credit == 0) {
                    return; // nothing hit Undeposited Funds
                }

                $data['debit_amount']    = $debit;
                $data['credit_amount']   = $credit;
                $data['je_description']  = $desc;
                $data['je_account_name'] = $acct;
                $data['je_line_name']    = $name;
                break;

            default:
                return;
        }

        // Upsert
        $existing = $this->where('qb_txn_id', $qbId)->first();
        if ($existing) {
            $this->update($existing['id'], $data);
        } else {
            $this->insert($data);
        }
    }

    /**
     * Pull by date range for the grid (only the `undeposited_funds` table).
     */
    public function fetchByDateRange(string $from, string $to): array
    {
        return $this->where('txn_date >=', $from)
                    ->where('txn_date <=', $to)
                    ->orderBy('txn_date', 'ASC')
                    ->findAll();
    }

    // -----------------------------------------------------------------------
    // Deposits portion (formerly DepositsMadeModel)
    // -----------------------------------------------------------------------

    /**
     * Upsert a single Deposit object from QBO into
     * `deposits_made` + `deposits_made_lines`.
     */
    public function upsertDeposit(array $depositData): void
    {
        $depId = $depositData['Id'] ?? null;
        if (! $depId) {
            // skip if no QBO ID
            return;
        }
        $depDate      = $depositData['TxnDate']    ?? null;
        $docNumber    = $depositData['DocNumber']  ?? '';
        $depToAcctRef = $depositData['DepositToAccountRef']['value'] ?? '';
        $depToAcctNam = $depositData['DepositToAccountRef']['name']  ?? '';
        $depTotal     = $depositData['TotalAmt']   ?? 0;
        $depMemo      = $depositData['PrivateNote'] ?? '';

        // connect DB
        $db = db_connect();

        // Check if deposit row exists in `deposits_made`
        $existing = $db->table('deposits_made')
                       ->where('qb_deposit_id', $depId)
                       ->get()
                       ->getRowArray();

        $headerData = [
            'qb_deposit_id'                => $depId,
            'deposit_date'                 => $depDate,
            'doc_number'                   => $docNumber,
            'deposit_to_account_ref_value' => $depToAcctRef,
            'deposit_to_account_ref_name'  => $depToAcctNam,
            'total_amt'                    => $depTotal,
            'memo'                         => $depMemo,
        ];

        if ($existing) {
            // update existing deposit header
            $db->table('deposits_made')
               ->where('id', $existing['id'])
               ->update($headerData);

            $headerId = $existing['id'];
        } else {
            // insert new deposit header
            $db->table('deposits_made')->insert($headerData);
            $headerId = $db->insertID();
        }

        // Remove old lines for this deposit, then insert fresh lines
        $db->table('deposits_made_lines')
           ->where('deposits_made_id', $headerId)
           ->delete();

        // Lines
        foreach ($depositData['Line'] ?? [] as $line) {
            $lineNum    = $line['Id']          ?? '';
            $lineDesc   = $line['Description'] ?? '';
            $lineAmount = $line['Amount']      ?? 0;
            $linkedTxns = $line['LinkedTxn']   ?? [];

            // If there are no LinkedTxns, we still store one line
            if (empty($linkedTxns)) {
                $this->insertLineRow(
                    $db, $headerId, $lineNum, $lineDesc, $lineAmount, '', ''
                );
            } else {
                // For each LinkedTxn, insert a new line
                foreach ($linkedTxns as $lt) {
                    $this->insertLineRow(
                        $db,
                        $headerId,
                        $lineNum,
                        $lineDesc,
                        $lineAmount,
                        ($lt['TxnType'] ?? ''),
                        ($lt['TxnId']   ?? '')
                    );
                }
            }
        }
    }

    /**
     * Insert a single row into `deposits_made_lines`.
     */
    private function insertLineRow(
        $db,
        int $headerId,
        string $lineNum,
        string $description,
        float $amount,
        string $txnType,
        string $txnId
    ): void {
        $db->table('deposits_made_lines')->insert([
            'deposits_made_id' => $headerId,
            'line_num'         => $lineNum,
            'description'      => $description,
            'amount'           => $amount,
            'linked_txn_type'  => $txnType,
            'linked_txn_id'    => $txnId,
        ]);
    }
}
