console.log("manage_undeposited_funds.js loaded.");

document.addEventListener("DOMContentLoaded", function() {
    const selectEl = document.querySelector('select[name="view_mode"]');
    const middlePane = document.getElementById('middlePane');
    const leftPane = document.getElementById('leftPane');

    if (!selectEl || !middlePane || !leftPane) return;

    // Restore view mode from sessionStorage
    const savedViewMode = sessionStorage.getItem('currentViewMode');
    if (savedViewMode && ['manage', 'pull', 'make_bank_deposit'].includes(savedViewMode)) {
        selectEl.value = savedViewMode;
    }

    // Helper to update Pull QB Data wrapper visibility
    function updateWrapperVisibility() {
        const pullQBDataWrapper = leftPane.querySelector('#pullQBDataWrapper');
        if (pullQBDataWrapper) {
            if (selectEl.value === 'manage') {
                pullQBDataWrapper.style.display = 'block';
            } else {
                pullQBDataWrapper.style.display = 'none';
            }
        }
    }

    // Handle select change (dropdown)
    selectEl.addEventListener('change', async () => {
        sessionStorage.setItem('currentViewMode', selectEl.value);
        await loadView(selectEl.value);
        await loadLeftPane(selectEl.value);
    });

    // AJAX: Load partial into middle pane
    async function loadView(mode, fresh = false, extraParams = '') {
        try {
            let url = fresh
                ? `/undepositedfunds/loadView?pane=middle&view=${mode}&fresh=1`
                : `/undepositedfunds/loadView?pane=middle&view=${mode}`;
            if (extraParams) {
                url += `&${extraParams.replace(/^\?/, '')}`;
            }

            const response = await fetch(url);
            if (!response.ok) {
                throw new Error(`Network response was not ok: ${response.statusText}`);
            }
            const html = await response.text();
            middlePane.innerHTML = html;

            attachPaginationClick(mode);
            if (mode === 'make_bank_deposit') {
                attachMakeBankDepositLogic();
            }
            attachCloseMakeDepositBtn();
        } catch (err) {
            console.error(`AJAX loadView ${mode} error:`, err);
            middlePane.innerHTML = `<p class="text-danger">Error loading view: ${mode}</p>`;
        }
    }

    // AJAX: Load partial into left pane
    async function loadLeftPane(mode) {
        try {
            let view = 'main_view_selector';
            if (mode === 'make_bank_deposit') {
                view = 'make_bank_deposit';
            }
            const url = `/undepositedfunds/loadView?pane=left&view=${view}`;
            const response = await fetch(url);
            if (!response.ok) {
                throw new Error(`Network response was not ok: ${response.statusText}`);
            }
            const html = await response.text();
            leftPane.innerHTML = html;
            attachLeftPaneButtonListeners();
            updateWrapperVisibility();
        } catch (err) {
            console.error(`AJAX loadLeftPane ${mode} error:`, err);
            leftPane.innerHTML = `<p class="text-danger">Error loading left pane: ${mode}</p>`;
        }
    }

    // First load: Restore view mode, or default to manage
    if (savedViewMode && savedViewMode !== 'manage') {
        loadView(savedViewMode);
        loadLeftPane(savedViewMode);
    } else {
        attachPaginationClick(selectEl.value);
        loadLeftPane(selectEl.value);
    }

    // --- START: Live Update Selected Amount Logic ---
    function updateSelectedAmount() {
        const checkboxes = middlePane ? middlePane.querySelectorAll('input[type="checkbox"][name="depositCheckbox[]"]:checked') : [];
        let sum = 0;
        checkboxes.forEach(box => {
            const amountAttr = box.getAttribute('data-amount');
            let amount = parseFloat(amountAttr);
            if (isNaN(amount)) amount = 0;
            sum += amount;
        });

        const display = leftPane ? leftPane.querySelector('#amountSelectedDisplay') : null;

        if (display) {
            const formatted = sum < 0
                ? `-\$${Math.abs(sum).toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}`
                : `\$${sum.toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}`;
            display.textContent = formatted;
        }
    }
    // --- END: Live Update Selected Amount Logic ---

    // Attach checkbox logic for make deposit view
    function attachMakeBankDepositLogic() {
        const checkboxes = middlePane.querySelectorAll('input[type="checkbox"][name="depositCheckbox[]"]');
        checkboxes.forEach(checkbox => {
            checkbox.addEventListener('change', async (e) => {
                const checked = e.target.checked;
                const ufId = e.target.value;
                try {
                    const res = await fetch('/undepositedfunds/updateSelected', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-Requested-With': 'XMLHttpRequest'
                        },
                        body: JSON.stringify({ ufId, checked })
                    });
                    if (!res.ok) throw new Error(`Failed to update selection: ${res.statusText}`);
                } catch (err) {
                    console.error('Error updating selection:', err);
                }
                updateSelectedAmount();
            });
        });
        updateSelectedAmount();
    }

    // Attach pagination click handling
    function attachPaginationClick(currentMode) {
        const paginationLinks = middlePane.querySelectorAll('.pagination a');
        paginationLinks.forEach(link => {
            link.addEventListener('click', async (evt) => {
                evt.preventDefault();
                const href = link.getAttribute('href') || '';
                const params = href.replace(/^\?/, '');
                await loadView(currentMode, false, params);
            });
        });
    }

    // Attach listeners for left pane buttons
    function attachLeftPaneButtonListeners() {
        const pullQBDataBtn = leftPane.querySelector('#pullQBDataBtn');
        const makeBankDepositBtn = leftPane.querySelector('#makeBankDepositBtn');
        const dateSelectorContainer = leftPane.querySelector('#dateSelectorContainer');

        // "Pull QB Data" button logic (injects date selector)
        if (pullQBDataBtn) {
            pullQBDataBtn.addEventListener('click', () => {
                if (dateSelectorContainer.innerHTML.trim() === '') {
                    // Inject date selector HTML directly
                    const today = new Date().toISOString().split('T')[0];
                    const firstOfMonth = new Date(new Date().getFullYear(), new Date().getMonth(), 1).toISOString().split('T')[0];
                    dateSelectorContainer.innerHTML = `
                        <div class="form-row flex-form">
                            <div class="col-sm-2">
                                <label>From Date</label>
                                <input type="date" name="from_date" value="${firstOfMonth}" class="form-control" required>
                            </div>
                            <div class="col-sm-2">
                                <label>To Date</label>
                                <input type="date" name="to_date" value="${today}" class="form-control" required>
                            </div>
                            <div class="col-sm-2">
                                <button type="button" id="confirmPullBtn" class="btn btn-primary w-100">Confirm Pull</button>
                            </div>
                        </div>
                    `;
                    // Attach the confirm button logic directly
                    if (!attachConfirmPullBtn(dateSelectorContainer)) {
                        setTimeout(() => attachConfirmPullBtn(dateSelectorContainer), 100);
                    }
                } else {
                    dateSelectorContainer.innerHTML = ''; // Toggle off if already loaded
                }
            });
        }

        // "Make Bank Deposit" button logic
        if (makeBankDepositBtn) {
            makeBankDepositBtn.addEventListener('click', async () => {
                sessionStorage.setItem('currentViewMode', 'make_bank_deposit');
                await loadView('make_bank_deposit', true);
                await loadLeftPane('make_bank_deposit');
            });
        }
        // Attach cancel (red X) after loading left pane
        attachCloseMakeDepositBtn();
        updateWrapperVisibility();
    }

    // Confirm Pull button handler (date selector)
    function attachConfirmPullBtn(dateSelectorContainer) {
        const confirmBtn = dateSelectorContainer.querySelector('#confirmPullBtn');
        const fromDate = dateSelectorContainer.querySelector('input[name="from_date"]');
        const toDate = dateSelectorContainer.querySelector('input[name="to_date"]');
        if (!confirmBtn || !fromDate || !toDate) {
            console.warn('ConfirmPullBtn or date inputs not found in dateSelectorContainer');
            return false;
        }

        // Remove any existing listeners to prevent duplicates
        confirmBtn.removeEventListener('click', handleConfirmPull);
        confirmBtn.addEventListener('click', handleConfirmPull);

        function handleConfirmPull() {
            if (!fromDate.value || !toDate.value) {
                alert('Please select both From Date and To Date');
                return;
            }
            fetch('/undepositedfunds', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                    'X-Requested-With': 'XMLHttpRequest'
                },
                body: new URLSearchParams({
                    action: 'pull_undeposited',
                    from_date: fromDate.value,
                    to_date: toDate.value
                })
            })
            .then(response => {
                if (!response.ok) throw new Error(`Network error: ${response.statusText}`);
                return response.text();
            })
            .then(data => {
                window.location.reload();
            })
            .catch(err => {
                alert('Error pulling data: ' + err.message);
            });
        }
        return true;
    }

    // Red X/cancel handler for make deposit view (ALWAYS attach directly from leftPane)
    function attachCloseMakeDepositBtn() {
        const closeBtn = leftPane.querySelector('#closeMakeDepositBtn');
        if (!closeBtn) return;
        closeBtn.addEventListener('click', async () => {
            console.log('RED X CLICKED'); // Debug: remove this line if not needed
            await fetch('/undepositedfunds/clearSelected', { method: 'POST' });
            sessionStorage.setItem('currentViewMode', 'manage');
            await loadView('manage', true);
            await loadLeftPane('manage');
            if (selectEl) selectEl.value = 'manage';
            updateWrapperVisibility();
        });
    }

    // If the current default is make_bank_deposit, load it
    if (selectEl.value === 'make_bank_deposit') {
        loadView('make_bank_deposit');
        loadLeftPane('make_bank_deposit');
    }
});
